<?php
/**
 * Process functions
 *
 * @package WPVulnerability
 *
 * @version 2.0.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Load the settings to be available always.
 *
 * @version 2.0.0
 *
 * @return array|false An array containing the WPVulnerability settings if they exist, or false if they don't.
 */
$wpvulnerability_settings = get_site_option( 'wpvulnerability-config' );
$wpvulnerability_analyze  = get_site_option( 'wpvulnerability-analyze' );

/**
 * Enqueues the WPVulnerability admin CSS file on WPVulnerability admin pages.
 *
 * @version 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_enqueue_scripts() {

	wp_enqueue_style(
		'wpvulnerability-admin',
		WPVULNERABILITY_PLUGIN_URL . 'assets/admin.css',
		array(),
		WPVULNERABILITY_PLUGIN_VERSION
	);
}
add_action( 'admin_enqueue_scripts', 'wpvulnerability_admin_enqueue_scripts' );

/**
 * Create the WP-Admin options page
 * This function generates the HTML output for the WPVulnerability settings page in the WP-Admin.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_create_admin_page() {

	/**
	 * Processes the form submission for the WPVulnerability plugin settings in a multisite network admin context.
	 *
	 * Checks if the current request is a submission from the WPVulnerability settings page.
	 * Verifies the security nonce to prevent CSRF attacks. If the verification fails or if the request
	 * is not from a network admin or the admin dashboard, the function will halt execution and display an error.
	 * Otherwise, it sanitizes and updates the plugin settings, reschedules any relevant wp-cron events,
	 * and registers a settings error to notify the user that the information has been updated.
	 *
	 * @since 3.0.0
	*/
	if ( isset( $_GET['page'] ) && 'wpvulnerability-options' === $_GET['page'] && ! empty( $_POST ) ) {

		if ( check_admin_referer( 'wpvulnerability_nonce', 'wpauto_nonce' ) ) {

			if ( isset( $_POST['wpvulnerability_submit'] ) && isset( $_POST['wpvulnerability-config'] ) ) {

				$wpvulnerability_sanitized_values['emails'] = null;
				$wpvulnerability_sanitized_values['period'] = 'weekly';

				$wpvulnerability_input['emails'] = array();
				if ( isset( $_POST['wpvulnerability-config']['emails'] ) ) {
					$wpvulnerability_config_emails    = wp_kses( wp_unslash( (string) $_POST['wpvulnerability-config']['emails'] ), 'strip' );
					$wpvulnerability_input_email_text = explode( ',', $wpvulnerability_config_emails );
					foreach ( $wpvulnerability_input_email_text as $wpvulnerability_input_email ) {
						$wpvulnerability_input_email = sanitize_email( trim( $wpvulnerability_input_email ) );
						if ( is_email( $wpvulnerability_input_email ) ) {
							$wpvulnerability_input['emails'][] = $wpvulnerability_input_email;
						}
						unset( $wpvulnerability_input_email );
					}
					unset( $wpvulnerability_input_email_text, $wpvulnerability_config_emails );
				}
				if ( count( $wpvulnerability_input['emails'] ) ) {
					$wpvulnerability_sanitized_values['emails'] = implode( ',', $wpvulnerability_input['emails'] );
				} else {
					$wpvulnerability_sanitized_values['emails'] = null;
				}

				$wpvulnerability_input['period'] = null;
				if ( isset( $_POST['wpvulnerability-config']['period'] ) ) {
					$wpvulnerability_input['period'] = wp_kses( wp_unslash( (string) $_POST['wpvulnerability-config']['period'] ), 'strip' );
				}
				switch ( $wpvulnerability_input['period'] ) {
					case 'daily':
						$wpvulnerability_sanitized_values['period'] = 'daily';
						break;
					case 'weekly':
					default:
						$wpvulnerability_sanitized_values['period'] = 'weekly';
						break;
				}
				unset( $wpvulnerability_input );

				update_site_option(
					'wpvulnerability-config',
					array(
						'emails' => $wpvulnerability_sanitized_values['emails'],
						'period' => $wpvulnerability_sanitized_values['period'],
					)
				);

				$current_schedule = wp_get_schedule( 'wpvulnerability_notification' );
				if ( $current_schedule !== $wpvulnerability_sanitized_values['period'] ) {
					wp_clear_scheduled_hook( 'wpvulnerability_notification' );
					wp_schedule_event( time(), $wpvulnerability_sanitized_values['period'], 'wpvulnerability_notification' );
				}
				unset( $current_schedule );

				unset( $wpvulnerability_sanitized_values );

				add_settings_error(
					'wpvulnerability-messages',
					'wpvulnerability-updated',
					__( 'Settings saved.', 'wpvulnerability' ),
					'success'
				);

			}

			if ( isset( $_POST['wpvulnerability_submit'] ) && isset( $_POST['wpvulnerability-analyze'] ) ) {

				$wpvulnerability_sanitized_values = array(
					'core'    => 0,
					'plugins' => 0,
					'themes'  => 0,
					'php'     => 0,
					'apache'  => 0,
					'nginx'   => 0,
					'mariadb' => 0,
					'mysql'   => 0,
				);

				$wpvulnerability_values = array_map( 'sanitize_text_field', wp_unslash( $_POST['wpvulnerability-analyze'] ) );

				foreach ( $wpvulnerability_values as $data ) {
					switch ( $data ) {
						case 'core':
							$wpvulnerability_sanitized_values['core'] = 1;
							break;
						case 'plugins':
							$wpvulnerability_sanitized_values['plugins'] = 1;
							break;
						case 'themes':
							$wpvulnerability_sanitized_values['themes'] = 1;
							break;
						case 'php':
							$wpvulnerability_sanitized_values['php'] = 1;
							break;
						case 'apache':
							$wpvulnerability_sanitized_values['apache'] = 1;
							break;
						case 'nginx':
							$wpvulnerability_sanitized_values['nginx'] = 1;
							break;
						case 'mariadb':
							$wpvulnerability_sanitized_values['mariadb'] = 1;
							break;
						case 'mysql':
							$wpvulnerability_sanitized_values['mysql'] = 1;
							break;
					}
				}

				update_site_option(
					'wpvulnerability-analyze',
					array(
						'core'    => $wpvulnerability_sanitized_values['core'],
						'plugins' => $wpvulnerability_sanitized_values['plugins'],
						'themes'  => $wpvulnerability_sanitized_values['themes'],
						'php'     => $wpvulnerability_sanitized_values['php'],
						'apache'  => $wpvulnerability_sanitized_values['apache'],
						'nginx'   => $wpvulnerability_sanitized_values['nginx'],
						'mariadb' => $wpvulnerability_sanitized_values['mariadb'],
						'mysql'   => $wpvulnerability_sanitized_values['mysql'],
					)
				);

				unset( $wpvulnerability_sanitized_values );

				add_settings_error(
					'wpvulnerability-messages',
					'wpvulnerability-updated',
					__( 'Settings saved.', 'wpvulnerability' ),
					'success'
				);

			}
		}
	}

	/**
	 * Reset the data
	 *
	 * @since 3.0.0
	 */
	if ( isset( $_POST['wpvulnerability_reset'] ) && check_admin_referer( 'wpvulnerability_nonce', 'wpauto_nonce' ) ) {

		// Calls the reset function.
		wpvulnerability_update_database_data();

		set_transient( 'wpvulnerability_message_manual_success', __( 'Data from source has been reloaded.', 'wpvulnerability' ), 10 );

	}

	/**
	 * Send an test email
	 *
	 * @since 3.0.0
	 */
	if ( isset( $_POST['wpvulnerability_email'] ) && check_admin_referer( 'wpvulnerability_nonce', 'wpauto_nonce' ) ) {

		if ( ! function_exists( 'wpvulnerability_execute_notification' ) ) {
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-core.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-plugins.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-themes.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-php.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-process.php';
			require_once WPVULNERABILITY_PLUGIN_PATH . '/wpvulnerability-notifications.php';
		}

		// Calls the notifications function, forced.
		$wpmail = wpvulnerability_execute_notification( true );

		if ( $wpmail ) {

			set_transient( 'wpvulnerability_message_manual_success', __( 'Test email has been sent.', 'wpvulnerability' ), 10 );

		} else {

			set_transient( 'wpvulnerability_message_manual_error', __( 'Test email has failed. Please, check your email settings.', 'wpvulnerability' ), 10 );

		}
	}

	$wpvulnerability_settings = get_site_option( 'wpvulnerability-config' );
	?>
	<div class="header-wrap">
		<div class="wrapper">
			<div class="header wpvulnerability-header">
				<div class="logo">
					<img src="<?php echo esc_url( WPVULNERABILITY_PLUGIN_URL ); ?>assets/logo64.png" style="height: 64px; vertical-align: text-top; width: 64px;" alt="" title="WPVulnerability">
					<h2><?php esc_html_e( 'WPVulnerability settings', 'wpvulnerability' ); ?></h2>
				</div>
			</div>
		</div>
	</div>
	<?php
	$wpvulnerability_message_manual_success = get_transient( 'wpvulnerability_message_manual_success' );
	if ( $wpvulnerability_message_manual_success ) {
		echo '<div class="notice notice-success"><p>' . esc_html( (string) $wpvulnerability_message_manual_success ) . '</p></div>';
		delete_transient( 'wpvulnerability_message_manual_success' );
		unset( $wpvulnerability_message_manual_success );
	}
	$wpvulnerability_message_manual_error = get_transient( 'wpvulnerability_message_manual_error' );
	if ( $wpvulnerability_message_manual_error ) {
		echo '<div class="notice notice-error"><p>' . esc_html( (string) $wpvulnerability_message_manual_error ) . '</p></div>';
		delete_transient( 'wpvulnerability_message_manual_error' );
		unset( $wpvulnerability_message_manual_error );
	}
	?>
	<?php settings_errors( 'wpvulnerability-messages', true ); ?>
	<div class="wrap">
	<div class="wpvulnerability-container">
	<div class="wpvulnerability-column">
		<section class="section">
			<form method="post">
				<?php
					settings_fields( 'admin_wpvulnerability_settings' );
					do_settings_sections( 'wpvulnerability-config' );
					wp_nonce_field( 'wpvulnerability_nonce', 'wpauto_nonce' );
					submit_button(
						__( 'Save settings', 'wpvulnerability' ),
						'primary',
						'wpvulnerability_submit'
					);
				?>
			</form>
		</section>
		<section class="section">
			<form method="post">
				<?php
					settings_fields( 'admin_wpvulnerability_analyze' );
					do_settings_sections( 'wpvulnerability-analyze' );
					wp_nonce_field( 'wpvulnerability_nonce', 'wpauto_nonce' );
					submit_button(
						__( 'Save settings', 'wpvulnerability' ),
						'primary',
						'wpvulnerability_submit'
					);
				?>
			</form>
		</section>
		<section class="section">
			<header class="section-header">
				<hr>
				<h3><?php esc_html_e( 'Reload the data from source', 'wpvulnerability' ); ?></h3>
			</header>
			<div class="section-content">
				<p><?php esc_html_e( 'Reload all Core, Plugins, Themes and other components information directly from the API to have updated data.', 'wpvulnerability' ); ?></p>
				<form method="post" action="settings.php?page=wpvulnerability-options">
					<?php wp_nonce_field( 'wpvulnerability_nonce', 'wpauto_nonce' ); ?>
					<input type="submit" name="wpvulnerability_reset" value="<?php esc_attr_e( 'Reload Data', 'wpvulnerability' ); ?>" class="button button-secondary">
				</form>
			</div>
		</section>
		<section class="section">
			<header class="section-header">
				<hr>
				<h3><?php esc_html_e( 'Email test', 'wpvulnerability' ); ?></h3>
			</header>
			<div class="section-content">
	<?php
		$from_email = null;
	if ( defined( 'WPVULNERABILITY_MAIL' ) ) {
		$from_email = sanitize_email( trim( WPVULNERABILITY_MAIL ) );
		if ( is_email( $from_email ) ) {
			?>
				<p><?php esc_html_e( 'The mail will be sent from (set on WPVULNERABILITY_MAIL): ', 'wpvulnerability' ); ?> <code><?php echo esc_html( $from_email ); ?></code></p>
			<?php
		}
	}
	if ( ! $from_email ) {
		$from_email = get_site_option( 'admin_email' );
		?>
				<p><?php esc_html_e( 'The mail will be sent from: ', 'wpvulnerability' ); ?> <code><?php echo esc_html( $from_email ); ?></code></p>
		<?php
	}
	?>
				<p><?php esc_html_e( 'Send an email with the vulnerabilities (or empty).', 'wpvulnerability' ); ?></p>
				<form method="post" action="settings.php?page=wpvulnerability-options">
					<?php wp_nonce_field( 'wpvulnerability_nonce', 'wpauto_nonce' ); ?>
					<input type="submit" name="wpvulnerability_email" value="<?php esc_attr_e( 'Send email', 'wpvulnerability' ); ?>" class="button button-secondary">
				</form>
			</div>
		</section>
		</div><div class="wpvulnerability-column">
	<?php
	$wpvulnerability_statistics = json_decode( get_site_option( 'wpvulnerability-statistics' ), true );
	?>
		<section class="section">
			<header class="section-header">
				<h2><?php esc_html_e( 'WPVulnerability Statistics', 'wpvulnerability' ); ?></h2>
			</header>
			<div class="section-content">
				<dl>
					<dt><?php esc_html_e( 'Plugins', 'wpvulnerability' ); ?></dt>
	<?php
	if ( isset( $wpvulnerability_statistics['plugins'] ) ) {
		?>
					<dd>
		<?php

		printf(
			// translators: number of vulnerabilities.
			esc_html( _n( '%s vulnerability', '%s vulnerabilities', absint( $wpvulnerability_statistics['plugins']['vulnerabilities'] ), 'wpvulnerability' ) ),
			esc_html( number_format_i18n( absint( $wpvulnerability_statistics['plugins']['vulnerabilities'] ) ) )
		);

		printf(
			// translators: number of plugins.
			esc_html( _n( ' (%s plugin)', ' (%s plugins)', absint( $wpvulnerability_statistics['plugins']['products'] ), 'wpvulnerability' ) ),
			esc_html( number_format_i18n( absint( $wpvulnerability_statistics['plugins']['products'] ) ) )
		);
		?>
					</dd>
		<?php
	} else {
		?>
					<dd><?php esc_html_e( 'Data not available.', 'wpvulnerability' ); ?></dd>
		<?php
	}
	?>
					<dt><?php esc_html_e( 'Themes', 'wpvulnerability' ); ?></dt>
	<?php
	if ( isset( $wpvulnerability_statistics['themes'] ) ) {
		?>
					<dd>
		<?php
		printf(
			// translators: number of vulnerabilities.
			esc_html( _n( '%s vulnerability', '%s vulnerabilities', absint( $wpvulnerability_statistics['themes']['vulnerabilities'] ), 'wpvulnerability' ) ),
			esc_html( number_format_i18n( absint( $wpvulnerability_statistics['themes']['vulnerabilities'] ) ) )
		);

		printf(
			// translators: number of themes.
			esc_html( _n( ' (%s theme)', ' (%s themes)', absint( $wpvulnerability_statistics['themes']['products'] ), 'wpvulnerability' ) ),
			esc_html( number_format_i18n( absint( $wpvulnerability_statistics['themes']['products'] ) ) )
		);
		?>
					</dd>
		<?php
	} else {
		?>
					<dd><?php esc_html_e( 'Data not available.', 'wpvulnerability' ); ?></dd>
		<?php
	}
	?>
					<dt><?php esc_html_e( 'PHP', 'wpvulnerability' ); ?></dt>
	<?php
	if ( isset( $wpvulnerability_statistics['php'] ) ) {
		?>
					<dd>
					<?php
					printf(
						// translators: number of vulnerabilities.
						esc_html( _n( '%s vulnerability', '%s vulnerabilities', absint( $wpvulnerability_statistics['php']['vulnerabilities'] ), 'wpvulnerability' ) ),
						esc_html( number_format_i18n( absint( $wpvulnerability_statistics['php']['vulnerabilities'] ) ) )
					);
					?>
					</dd>
		<?php
	} else {
		?>
					<dd><?php esc_html_e( 'Data not available.', 'wpvulnerability' ); ?></dd>
		<?php
	}
	?>
					<dt><?php esc_html_e( 'Apache HTTPD', 'wpvulnerability' ); ?></dt>
	<?php
	if ( isset( $wpvulnerability_statistics['apache'] ) ) {
		?>
					<dd>
					<?php
					printf(
						// translators: number of vulnerabilities.
						esc_html( _n( '%s vulnerability', '%s vulnerabilities', absint( $wpvulnerability_statistics['apache']['vulnerabilities'] ), 'wpvulnerability' ) ),
						esc_html( number_format_i18n( absint( $wpvulnerability_statistics['apache']['vulnerabilities'] ) ) )
					);
					?>
					</dd>
		<?php
	} else {
		?>
					<dd><?php esc_html_e( 'Data not available.', 'wpvulnerability' ); ?></dd>
		<?php
	}
	?>
					<dt><?php esc_html_e( 'nginx', 'wpvulnerability' ); ?></dt>
	<?php
	if ( isset( $wpvulnerability_statistics['nginx'] ) ) {
		?>
					<dd>
					<?php
					printf(
						// translators: number of vulnerabilities.
						esc_html( _n( '%s vulnerability', '%s vulnerabilities', absint( $wpvulnerability_statistics['nginx']['vulnerabilities'] ), 'wpvulnerability' ) ),
						esc_html( number_format_i18n( absint( $wpvulnerability_statistics['nginx']['vulnerabilities'] ) ) )
					);
					?>
					</dd>
		<?php
	} else {
		?>
					<dd><?php esc_html_e( 'Data not available.', 'wpvulnerability' ); ?></dd>
		<?php
	}
	?>
					<dt><?php esc_html_e( 'MariaDB', 'wpvulnerability' ); ?></dt>
	<?php
	if ( isset( $wpvulnerability_statistics['mariadb'] ) ) {
		?>
					<dd>
					<?php
					printf(
						// translators: number of vulnerabilities.
						esc_html( _n( '%s vulnerability', '%s vulnerabilities', absint( $wpvulnerability_statistics['mariadb']['vulnerabilities'] ), 'wpvulnerability' ) ),
						esc_html( number_format_i18n( absint( $wpvulnerability_statistics['mariadb']['vulnerabilities'] ) ) )
					);
					?>
					</dd>
		<?php
	} else {
		?>
					<dd><?php esc_html_e( 'Data not available.', 'wpvulnerability' ); ?></dd>
		<?php
	}
	?>
					<dt><?php esc_html_e( 'MySQL', 'wpvulnerability' ); ?></dt>
	<?php
	if ( isset( $wpvulnerability_statistics['mysql'] ) ) {
		?>
					<dd>
					<?php
					printf(
						// translators: number of vulnerabilities.
						esc_html( _n( '%s vulnerability', '%s vulnerabilities', absint( $wpvulnerability_statistics['mysql']['vulnerabilities'] ), 'wpvulnerability' ) ),
						esc_html( number_format_i18n( absint( $wpvulnerability_statistics['mysql']['vulnerabilities'] ) ) )
					);
					?>
					</dd>
		<?php
	} else {
		?>
					<dd><?php esc_html_e( 'Data not available.', 'wpvulnerability' ); ?></dd>
		<?php
	}
	?>
				</dl>
	<?php
	if ( isset( $wpvulnerability_statistics['updated'] ) ) {
		switch_to_locale( determine_locale() );
		$formatted_datetime = date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), (int) $wpvulnerability_statistics['updated']['unixepoch'] );
		?>
				<p><small>
				<?php
				// translators: date of last update.
				printf( esc_html__( 'Updated: %s', 'wpvulnerability' ), esc_html( $formatted_datetime ) );
				?>
				</small></p>
		<?php
		restore_previous_locale();
	}
	?>
			</div>
		</section>
		<section class="section">
			<header class="section-header">
				<h2><?php esc_html_e( 'Behind the Project', 'wpvulnerability' ); ?></h2>
			</header>
			<div class="section-content"><div class="wpvulnerability-container">
				<div class="wpvulnerability-column">
				<h3><?php esc_html_e( 'Sponsors', 'wpvulnerability' ); ?></h3>
	<?php
	if ( isset( $wpvulnerability_statistics['sponsors'] ) ) {
		foreach ( $wpvulnerability_statistics['sponsors'] as $sponsor ) {
			?>
				<p><img src="<?php echo esc_url( $sponsor['image'] ); ?>" width="32" height="32" style="vertical-align: middle; margin-right: 8px;"> <a href="<?php echo esc_url( $sponsor['url'] ); ?>" target="_blank" rel="noreferrer noopener"><?php echo esc_html( $sponsor['name'] ); ?></a></p>
			<?php
		}
	} else {
		?>
				<p><?php esc_html_e( 'Data not available.', 'wpvulnerability' ); ?></p>
		<?php
	}
	?>

				</div><div class="wpvulnerability-column">

				<h3><?php esc_html_e( 'Contributors', 'wpvulnerability' ); ?></h3>
	<?php
	if ( isset( $wpvulnerability_statistics['contributors'] ) ) {
		foreach ( $wpvulnerability_statistics['contributors'] as $contributor ) {
			?>
				<p><img src="<?php echo esc_url( $contributor['image'] ); ?>" width="32" height="32" style="vertical-align: middle; margin-right: 8px;"> <a href="<?php echo esc_url( $contributor['url'] ); ?>" target="_blank" rel="noreferrer noopener"><?php echo esc_html( $contributor['name'] ); ?></a></p>
			<?php
		}
	} else {
		?>
				<p><?php esc_html_e( 'Data not available.', 'wpvulnerability' ); ?></p>
		<?php
	}
	?>

				</div>
			</div></div>
		</section>
		</div>
	</div></div>
	<?php
}

/**
 * Adds a WP-Admin menu option for the WPVulnerability plugin
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_menu() {

	// Adds a submenu page under the Settings menu.
	add_submenu_page(
		'settings.php',
		__( 'WPVulnerability', 'wpvulnerability' ),
		__( 'WPVulnerability', 'wpvulnerability' ),
		'manage_network_options',
		'wpvulnerability-options',
		'wpvulnerability_create_admin_page'
	);
}
add_action( 'network_admin_menu', 'wpvulnerability_admin_menu' );

/**
 * Print the settings header information for the notifications section.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_section_notifications() {

	// Output the header information for the notifications section.
	esc_html_e( 'Configure and save these settings to receive email notifications.', 'wpvulnerability' );
}

/**
 * Callback function to display the email input field in the admin settings page.
 * This function retrieves the current WPVulnerability plugin settings and displays the email input field
 * for users to enter their email addresses. If no email is saved in the settings, the admin email is displayed.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_emails_callback() {

	// Retrieve the WPVulnerability plugin settings.
	$wpvulnerability_settings = get_site_option( 'wpvulnerability-config' );

	// Set a default value for the email input field if no email is saved in the settings.
	if ( ! isset( $wpvulnerability_settings['emails'] ) ) {
		$wpvulnerability_settings['emails'] = '';
	}

	// Output the email input field. Use the network admin email as a placeholder in a multisite environment.
	$admin_email = get_site_option( 'admin_email' );

	// Output the email input field and display the admin email as a hint.
	?>
	<input class="regular-text" type="text" name="wpvulnerability-config[emails]" id="wpvulnerability_emails" placeholder="<?php echo esc_attr( (string) $admin_email ); ?>" value="<?php echo esc_attr( (string) $wpvulnerability_settings['emails'] ); ?>">
	<br><small><?php esc_html_e( 'Default administrator email', 'wpvulnerability' ); ?>: <?php echo esc_attr( (string) $admin_email ); ?></small>
	<?php

	unset( $admin_email );
}

/**
 * Print the settings header information for the analyze section.
 *
 * @since 3.3.0
 *
 * @return void
 */
function wpvulnerability_admin_section_analyze() {

	// Output the header information for the notifications section.
	esc_html_e( 'Configure and save these settings to hide vulnerabilities.', 'wpvulnerability' );
}

/**
 * Print when to send the vulnerability scan emails.
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_period_callback() {

	// Get the saved plugin settings.
	$wpvulnerability_settings = get_site_option( 'wpvulnerability-config', array() );

	// Set the default period to 'weekly' if not set or empty.
	if ( ! isset( $wpvulnerability_settings['period'] ) || empty( $wpvulnerability_settings['period'] ) ) {
		$wpvulnerability_settings['period'] = 'weekly';
	}

	// Output the period select box.
	?>
	<div id="wpvulnerability_period">
	<label>
		<input type="radio" name="wpvulnerability-config[period]" value="daily" <?php checked( $wpvulnerability_settings['period'], 'daily' ); ?> />
		<?php esc_html_e( 'Daily', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="radio" name="wpvulnerability-config[period]" value="weekly" <?php checked( $wpvulnerability_settings['period'], 'weekly' ); ?> />
		<?php esc_html_e( 'Weekly', 'wpvulnerability' ); ?>
	</label>
	</div>
	<?php
}

/**
 * Displays the WPVulnerability plugin analysis settings in the admin panel.
 *
 * This function retrieves the current WPVulnerability analysis settings and
 * ensures all necessary options are set. It then outputs a multiple-select
 * field allowing the user to select which components (core, plugins, themes,
 * php, apache, nginx) to analyze.
 *
 * @since 3.3.0
 *
 * @return void
 */
function wpvulnerability_admin_analyze_callback() {

	// Retrieve the WPVulnerability plugin settings.
	$wpvulnerability_analyze = get_site_option( 'wpvulnerability-analyze', array() );
	if ( ! isset( $wpvulnerability_analyze['core'] ) ) {
		$wpvulnerability_analyze['core'] = 0;
	}
	if ( ! isset( $wpvulnerability_analyze['plugins'] ) ) {
		$wpvulnerability_analyze['plugins'] = 0;
	}
	if ( ! isset( $wpvulnerability_analyze['themes'] ) ) {
		$wpvulnerability_analyze['themes'] = 0;
	}
	if ( ! isset( $wpvulnerability_analyze['php'] ) ) {
		$wpvulnerability_analyze['php'] = 0;
	}
	if ( ! isset( $wpvulnerability_analyze['apache'] ) ) {
		$wpvulnerability_analyze['apache'] = 0;
	}
	if ( ! isset( $wpvulnerability_analyze['nginx'] ) ) {
		$wpvulnerability_analyze['nginx'] = 0;
	}
	if ( ! isset( $wpvulnerability_analyze['mariadb'] ) ) {
		$wpvulnerability_analyze['mariadb'] = 0;
	}
	if ( ! isset( $wpvulnerability_analyze['mysql'] ) ) {
		$wpvulnerability_analyze['mysql'] = 0;
	}

	?>
	<div id="wpvulnerability_analyze">
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[core]" value="core" <?php checked( $wpvulnerability_analyze['core'] ); ?> />
		<?php esc_html_e( 'Core', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[plugins]" value="plugins" <?php checked( $wpvulnerability_analyze['plugins'] ); ?> />
		<?php esc_html_e( 'Plugins', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[themes]" value="themes" <?php checked( $wpvulnerability_analyze['themes'] ); ?> />
		<?php esc_html_e( 'Themes', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[php]" value="php" <?php checked( $wpvulnerability_analyze['php'] ); ?> />
		<?php esc_html_e( 'PHP', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[apache]" value="apache" <?php checked( $wpvulnerability_analyze['apache'] ); ?> />
		<?php esc_html_e( 'Apache HTTPD', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[nginx]" value="nginx" <?php checked( $wpvulnerability_analyze['nginx'] ); ?> />
		<?php esc_html_e( 'nginx', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[mariadb]" value="mariadb" <?php checked( $wpvulnerability_analyze['mariadb'] ); ?> />
		<?php esc_html_e( 'MariaDB', 'wpvulnerability' ); ?>
	</label>
	<br/>
	<label>
		<input type="checkbox" name="wpvulnerability-analyze[mysql]" value="mysql" <?php checked( $wpvulnerability_analyze['mysql'] ); ?> />
		<?php esc_html_e( 'MySQL', 'wpvulnerability' ); ?>
	</label>
	</div>
	<?php
}

/**
 * Content for the Dashboard Widget
 *
 * @since 2.2.0
 *
 * @return void
 */
function wpvulnerability_admin_dashboard_content() {
	// Get the number of core vulnerabilites from cache.
	$wpvulnerability_test_core_counter = json_decode( get_site_option( 'wpvulnerability-core-vulnerable' ) );
	if ( ! is_numeric( $wpvulnerability_test_core_counter ) ) {
		$wpvulnerability_test_core_counter = 0;
	}
	/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
	$msg_core = sprintf( _n( 'Core: %d vulnerability', 'Core: %d vulnerabilities', $wpvulnerability_test_core_counter, 'wpvulnerability' ), $wpvulnerability_test_core_counter );

	// Get the number of themes vulnerabilites from cache.
	$wpvulnerability_test_themes_counter = json_decode( get_site_option( 'wpvulnerability-themes-vulnerable' ) );
	if ( ! is_numeric( $wpvulnerability_test_themes_counter ) ) {
		$wpvulnerability_test_themes_counter = 0;
	}
	/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
	$msg_themes = sprintf( _n( 'Themes: %d vulnerability', 'Themes: %d vulnerabilities', $wpvulnerability_test_themes_counter, 'wpvulnerability' ), $wpvulnerability_test_themes_counter );

	// Get the number of plugins vulnerabilites from cache.
	$wpvulnerability_test_plugins_counter = json_decode( get_site_option( 'wpvulnerability-plugins-vulnerable' ) );
	if ( ! is_numeric( $wpvulnerability_test_plugins_counter ) ) {
		$wpvulnerability_test_plugins_counter = 0;
	}
	/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
	$msg_plugins = sprintf( _n( 'Plugins: %d vulnerability', 'Plugins: %d vulnerabilities', $wpvulnerability_test_plugins_counter, 'wpvulnerability' ), $wpvulnerability_test_plugins_counter );

	// Get the number of PHP vulnerabilites from cache.
	$wpvulnerability_test_php_counter = json_decode( get_site_option( 'wpvulnerability-php-vulnerable' ) );
	if ( ! is_numeric( $wpvulnerability_test_php_counter ) ) {
		$wpvulnerability_test_php_counter = 0;
	}
	$wpvulnerability_test_php_version = wpvulnerability_sanitize_version_php( phpversion() );
	/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
	$msg_php = sprintf( __( 'PHP %s: ', 'wpvulnerability' ), $wpvulnerability_test_php_version ) . sprintf( _n( '%d vulnerability', '%d vulnerabilities', $wpvulnerability_test_php_counter, 'wpvulnerability' ), $wpvulnerability_test_php_counter );

	// Get the number of Apache HTTPD vulnerabilites from cache.
	$show_apache = false;
	$msg_apache  = null;
	$webserver   = wpvulnerability_detect_webserver();
	if ( isset( $webserver['id'] ) && 'apache' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {
		// Get the Apache HTTPD version.
		$apache_version                      = wp_kses( (string) $webserver['version'], 'strip' );
		$wpvulnerability_test_apache_counter = json_decode( get_site_option( 'wpvulnerability-apache-vulnerable' ) );
		if ( ! is_numeric( $wpvulnerability_test_apache_counter ) ) {
			$wpvulnerability_test_apache_counter = 0;
		}
		$wpvulnerability_test_apache_version = wpvulnerability_sanitize_version_apache( $apache_version );
		$show_apache                         = true;
		/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
		$msg_apache = sprintf( __( 'Apache %s: ', 'wpvulnerability' ), $wpvulnerability_test_apache_version ) . sprintf( _n( '%d vulnerability', '%d vulnerabilities', $wpvulnerability_test_apache_counter, 'wpvulnerability' ), $wpvulnerability_test_apache_counter );
	}

	// Get the number of nginx vulnerabilites from cache.
	$show_nginx = false;
	$msg_nginx  = null;
	$webserver  = wpvulnerability_detect_webserver();
	if ( isset( $webserver['id'] ) && 'nginx' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {
		// Get the nginx version.
		$nginx_version                      = wp_kses( (string) $webserver['version'], 'strip' );
		$wpvulnerability_test_nginx_counter = json_decode( get_site_option( 'wpvulnerability-nginx-vulnerable' ) );
		if ( ! is_numeric( $wpvulnerability_test_nginx_counter ) ) {
			$wpvulnerability_test_nginx_counter = 0;
		}
		$wpvulnerability_test_nginx_version = wpvulnerability_sanitize_version_nginx( $nginx_version );
		$show_nginx                         = true;
		/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
		$msg_nginx = sprintf( __( 'nginx %s: ', 'wpvulnerability' ), $wpvulnerability_test_nginx_version ) . sprintf( _n( '%d vulnerability', '%d vulnerabilities', $wpvulnerability_test_nginx_counter, 'wpvulnerability' ), $wpvulnerability_test_nginx_counter );
	}

	// Get the number of mariadb vulnerabilites from cache.
	$show_mariadb = false;
	$msg_mariadb  = null;
	$sqlserver    = wpvulnerability_detect_sqlserver();
	if ( isset( $sqlserver['id'] ) && 'mariadb' === $sqlserver['id'] && isset( $sqlserver['version'] ) && $sqlserver['version'] ) {
		// Get the mariadb version.
		$mariadb_version                      = wp_kses( (string) $sqlserver['version'], 'strip' );
		$wpvulnerability_test_mariadb_counter = json_decode( get_site_option( 'wpvulnerability-mariadb-vulnerable' ) );
		if ( ! is_numeric( $wpvulnerability_test_mariadb_counter ) ) {
			$wpvulnerability_test_mariadb_counter = 0;
		}
		$wpvulnerability_test_mariadb_version = wpvulnerability_sanitize_version_mariadb( $mariadb_version );
		$show_mariadb                         = true;
		/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
		$msg_mariadb = sprintf( __( 'MariaDB %s: ', 'wpvulnerability' ), $wpvulnerability_test_mariadb_version ) . sprintf( _n( '%d vulnerability', '%d vulnerabilities', $wpvulnerability_test_mariadb_counter, 'wpvulnerability' ), $wpvulnerability_test_mariadb_counter );
	}

	// Get the number of mysql vulnerabilites from cache.
	$show_mysql = false;
	$msg_mysql  = null;
	$sqlserver  = wpvulnerability_detect_sqlserver();
	if ( isset( $sqlserver['id'] ) && 'mysql' === $sqlserver['id'] && isset( $sqlserver['version'] ) && $sqlserver['version'] ) {
		// Get the mysql version.
		$mysql_version                      = wp_kses( (string) $sqlserver['version'], 'strip' );
		$wpvulnerability_test_mysql_counter = json_decode( get_site_option( 'wpvulnerability-mysql-vulnerable' ) );
		if ( ! is_numeric( $wpvulnerability_test_mysql_counter ) ) {
			$wpvulnerability_test_mysql_counter = 0;
		}
		$wpvulnerability_test_mysql_version = wpvulnerability_sanitize_version_mysql( $mysql_version );
		$show_mysql                         = true;
		/* translators: Show the number of vulnerabilities in a WP-Admin dashboard */
		$msg_mysql = sprintf( __( 'MySQL %s: ', 'wpvulnerability' ), $wpvulnerability_test_mysql_version ) . sprintf( _n( '%d vulnerability', '%d vulnerabilities', $wpvulnerability_test_mysql_counter, 'wpvulnerability' ), $wpvulnerability_test_mysql_counter );
	}

	// Show the widget.
	echo esc_html( __( 'Vulnerability analysis of your WordPress installation:', 'wpvulnerability' ) );
	echo '<ul>';

	if ( wpvulnerability_analyze_filter( 'core' ) && ! $wpvulnerability_test_core_counter ) {
		echo '<li>✔️ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-wordpress.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_core ) . '</li>';
	} elseif ( wpvulnerability_analyze_filter( 'core' ) ) {
		echo '<li>❌ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-wordpress.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_core ) . '</li>';
	}

	if ( wpvulnerability_analyze_filter( 'plugins' ) && ! $wpvulnerability_test_plugins_counter ) {
		echo '<li>✔️ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-plugin.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_plugins ) . '</li>';
	} elseif ( wpvulnerability_analyze_filter( 'plugins' ) ) {
		echo '<li>❌ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-plugin.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_plugins );
		echo wpvulnerability_list_plugins(); // phpcs:ignore
		echo '</li>';
	}

	if ( wpvulnerability_analyze_filter( 'themes' ) && ! $wpvulnerability_test_themes_counter ) {
		echo '<li>✔️ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-theme.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_themes ) . '</li>';
	} elseif ( wpvulnerability_analyze_filter( 'themes' ) ) {
		echo '<li>❌ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-theme.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_themes );
		echo wpvulnerability_list_themes(); // phpcs:ignore
		echo '</li>';
	}

	if ( wpvulnerability_analyze_filter( 'php' ) && ! $wpvulnerability_test_php_counter ) {
		echo '<li>✔️ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-php.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_php ) . '</li>';
	} elseif ( wpvulnerability_analyze_filter( 'php' ) ) {
		echo '<li>❌ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-php.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_php ) . '</li>';
	}

	if ( $show_apache ) {
		if ( wpvulnerability_analyze_filter( 'apache' ) && ! $wpvulnerability_test_apache_counter ) {
			echo '<li>✔️ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-apache.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_apache ) . '</li>';
		} elseif ( wpvulnerability_analyze_filter( 'apache' ) ) {
			echo '<li>❌ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-apache.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_apache ) . '</li>';
		}
	}

	if ( $show_nginx ) {
		if ( wpvulnerability_analyze_filter( 'nginx' ) && ! $wpvulnerability_test_nginx_counter ) {
			echo '<li>✔️ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-nginx.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_nginx ) . '</li>';
		} elseif ( wpvulnerability_analyze_filter( 'nginx' ) ) {
			echo '<li>❌ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-nginx.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_nginx ) . '</li>';
		}
	}

	if ( $show_mariadb ) {
		if ( wpvulnerability_analyze_filter( 'mariadb' ) && ! $wpvulnerability_test_mariadb_counter ) {
			echo '<li>✔️ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-mariadb.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_mariadb ) . '</li>';
		} elseif ( wpvulnerability_analyze_filter( 'mariadb' ) ) {
			echo '<li>❌ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-mariadb.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_mariadb ) . '</li>';
		}
	}

	if ( $show_mysql ) {
		if ( wpvulnerability_analyze_filter( 'mysql' ) && ! $wpvulnerability_test_mysql_counter ) {
			echo '<li>✔️ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-mysql.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_mysql ) . '</li>';
		} elseif ( wpvulnerability_analyze_filter( 'mysql' ) ) {
			echo '<li>❌ <img src="' . esc_url( WPVULNERABILITY_PLUGIN_URL ) . 'assets/icon-mysql.svg" width="12" height="12" alt=""> ' . esc_html( (string) $msg_mysql ) . '</li>';
		}
	}

	echo '</ul>';
	echo esc_html( __( 'More information? Visit', 'wpvulnerability' ) ) . ' <a href="' . esc_url( get_admin_url( null, 'site-health.php' ) ) . '">' . esc_html( __( 'Site Health' ) ) . '</a>';

	unset( $wpvulnerability_test_core_counter, $wpvulnerability_test_themes_counter, $wpvulnerability_test_plugins_counter );
}

/**
 * Created a widget in the WordPress dashboard with vulnerability info.
 *
 * @since 2.2.0
 *
 * @return void
 */
function wpvulnerability_admin_dashboard() {

	if ( wpvulnerability_capabilities() ) {
		wp_add_dashboard_widget( 'wpvulnerability', __( 'WPVulnerability Status', 'wpvulnerability' ), 'wpvulnerability_admin_dashboard_content', null, null, 'side', 'high' );
	}
}
add_action( 'wp_network_dashboard_setup', 'wpvulnerability_admin_dashboard' );

/**
 * Initializes the WP-Admin settings page for the WP Vulnerability plugin
 *
 * @since 2.0.0
 *
 * @return void
 */
function wpvulnerability_admin_init() {

	// Register the plugin settings to be saved in the database.
	register_setting(
		'admin_wpvulnerability_settings',
		'wpvulnerability-config'
	);

	// Register messages.
	register_setting(
		'admin_wpvulnerability_settings',
		'wpvulnerability-messages'
	);

	// Add a section to the settings page.
	add_settings_section(
		'admin_wpvulnerability_settings',
		__( 'Receive notifications in your email', 'wpvulnerability' ),
		'wpvulnerability_admin_section_notifications',
		'wpvulnerability-config'
	);

	// Add a field to the settings page for the email addresses.
	add_settings_field(
		'wpvulnerability_emails',
		__( 'eMail addresses to notify (separated by commas)', 'wpvulnerability' ),
		'wpvulnerability_admin_emails_callback',
		'wpvulnerability-config',
		'admin_wpvulnerability_settings'
	);

	// Add a field to the settings page for the notification period.
	add_settings_field(
		'wpvulnerability_period',
		__( 'How often you want to receive notifications', 'wpvulnerability' ),
		'wpvulnerability_admin_period_callback',
		'wpvulnerability-config',
		'admin_wpvulnerability_settings'
	);

	// Register the plugin settings to be saved in the database.
	register_setting(
		'admin_wpvulnerability_analyze',
		'wpvulnerability-analyze'
	);

	// Add a section to the settings page.
	add_settings_section(
		'admin_wpvulnerability_analyze',
		__( 'Vulnerabilities to hide', 'wpvulnerability' ),
		'wpvulnerability_admin_section_analyze',
		'wpvulnerability-analyze'
	);

	// Add a field to the settings page for analyzing things.
	add_settings_field(
		'wpvulnerability_analyze',
		__( 'What do you want to hide?', 'wpvulnerability' ),
		'wpvulnerability_admin_analyze_callback',
		'wpvulnerability-analyze',
		'admin_wpvulnerability_analyze'
	);
}
add_action( 'admin_init', 'wpvulnerability_admin_init' );
