<?php
/**
 * Nginx functions
 *
 * @package WPVulnerability
 *
 * @version 3.2.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Retrieves vulnerabilities for an nginx version and updates its data.
 *
 * This function detects the installed nginx version, checks for vulnerabilities using an external API,
 * and updates the nginx data array with the vulnerabilities found.
 *
 * @since 3.2.0
 *
 * @return array The updated nginx data array containing vulnerability information.
 */
function wpvulnerability_get_fresh_nginx_vulnerabilities() {

	$nginx_version = null;
	$webserver     = wpvulnerability_detect_webserver();
	if ( isset( $webserver['id'] ) && 'nginx' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {
		// Get the nginx version.
		$nginx_version = wp_kses( (string) $webserver['version'], 'strip' );
	}

	// Initialize vulnerability related fields.
	$nginx_data['vulnerabilities'] = null;
	$nginx_data['vulnerable']      = 0;

	// Retrieve vulnerabilities for the nginx using its version.
	if ( $nginx_version ) {

		$nginx_api_response = wpvulnerability_get_nginx( $nginx_version, 0 );

		// If vulnerabilities are found, update the nginx data accordingly.
		if ( ! empty( $nginx_api_response ) ) {

			$nginx_data['vulnerabilities'] = $nginx_api_response;
			$nginx_data['vulnerable']      = 1;

		}
	}

	return $nginx_data;
}

/**
 * Get Installed nginx
 *
 * Retrieves the list of installed nginx versions, checks for vulnerabilities,
 * caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 3.2.0
 *
 * @return string JSON-encoded array of nginx data with vulnerabilities and vulnerable status.
 */
function wpvulnerability_nginx_get_installed() {

	$wpvulnerability_nginx_vulnerable = 0;

	// Retrieve fresh vulnerabilities for the installed nginx version.
	$nginx = wpvulnerability_get_fresh_nginx_vulnerabilities();

	// Check if the nginx version is vulnerable and count the vulnerabilities.
	if ( isset( $nginx['vulnerable'] ) && (int) $nginx['vulnerable'] ) {
		$wpvulnerability_nginx_vulnerable = count( $nginx['vulnerabilities'] );
	}

	// Cache the vulnerability data and the timestamp for cache expiration.
	if ( is_multisite() ) {
		update_site_option( 'wpvulnerability-nginx', wp_json_encode( $nginx ) );
		update_site_option( 'wpvulnerability-nginx-vulnerable', wp_json_encode( number_format( $wpvulnerability_nginx_vulnerable, 0, '.', '' ) ) );
		update_site_option( 'wpvulnerability-nginx-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	} else {
		update_option( 'wpvulnerability-nginx', wp_json_encode( $nginx ) );
		update_option( 'wpvulnerability-nginx-vulnerable', wp_json_encode( number_format( $wpvulnerability_nginx_vulnerable, 0, '.', '' ) ) );
		update_option( 'wpvulnerability-nginx-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	}

	// Return the JSON-encoded array of nginx data.
	return wp_json_encode( $nginx );
}

/**
 * Get the cached nginx vulnerabilities or update the cache if it's stale or missing.
 *
 * This function checks the cache for stored nginx vulnerabilities. If the cache is stale or missing,
 * it updates the cache with the latest vulnerabilities data.
 *
 * @since 3.2.0
 *
 * @return array Array of nginx data with their vulnerabilities.
 */
function wpvulnerability_nginx_get_vulnerabilities() {

	if ( is_multisite() ) {

		// Get the cached nginx data and decode it.
		$nginx_data_cache = json_decode( get_site_option( 'wpvulnerability-nginx-cache' ) );

		// Get the installed nginx data and decode it.
		$nginx_data = json_decode( get_site_option( 'wpvulnerability-nginx' ), true );

	} else {

		// Get the cached nginx data and decode it.
		$nginx_data_cache = json_decode( get_option( 'wpvulnerability-nginx-cache' ) );

		// Get the installed nginx data and decode it.
		$nginx_data = json_decode( get_option( 'wpvulnerability-nginx' ), true );

	}

	// If the cache is stale or the nginx data is empty, update the cache.
	if ( $nginx_data_cache < time() || empty( $nginx_data ) ) {

		// Get the installed nginx data and update the cache.
		$nginx_data = json_decode( wpvulnerability_nginx_get_installed(), true );

	}

	return $nginx_data;
}

/**
 * Update the nginx cache and remove any old cache data.
 *
 * This function refreshes the cache for nginx vulnerabilities by calling the function
 * that retrieves and updates the installed nginx data, ensuring that the cache contains
 * the most recent information.
 *
 * @since 3.2.0
 *
 * @return void
 */
function wpvulnerability_nginx_get_vulnerabilities_clean() {

	// Update the installed nginx cache.
	wpvulnerability_nginx_get_installed();
}
