<?php
/**
 * MariaDB functions
 *
 * @package WPVulnerability
 *
 * @version 3.4.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Retrieves vulnerabilities for a MariaDB version and updates its data.
 *
 * This function detects the installed SQL server type and version. If MariaDB is detected,
 * it checks for vulnerabilities using an external API and updates the MariaDB data array
 * with the vulnerabilities found.
 *
 * @since 3.4.0
 *
 * @return array The updated MariaDB data array containing vulnerability information.
 */
function wpvulnerability_get_fresh_mariadb_vulnerabilities() {

	$mariadb_version = null;
	$sqlserver       = wpvulnerability_detect_sqlserver();

	// Check if the detected SQL server is MariaDB and has a valid version.
	if ( isset( $sqlserver['id'] ) && 'mariadb' === $sqlserver['id'] && ! empty( $sqlserver['version'] ) ) {
		// Get the MariaDB version and sanitize it.
		$mariadb_version = wp_kses( (string) $sqlserver['version'], 'strip' );
	}

	// Initialize vulnerability-related fields in the MariaDB data array.
	$mariadb_data['vulnerabilities'] = null;
	$mariadb_data['vulnerable']      = 0;

	// If the MariaDB version is available, retrieve vulnerabilities for it.
	if ( $mariadb_version ) {

		$mariadb_api_response = wpvulnerability_get_mariadb( $mariadb_version, 0 );

		// If vulnerabilities are found, update the MariaDB data accordingly.
		if ( ! empty( $mariadb_api_response ) ) {

			$mariadb_data['vulnerabilities'] = $mariadb_api_response;
			$mariadb_data['vulnerable']      = 1;

		}
	}

	return $mariadb_data;
}

/**
 * Retrieve and Cache Installed MariaDB Vulnerabilities
 *
 * This function retrieves the list of installed MariaDB versions, checks for vulnerabilities,
 * caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 3.4.0
 *
 * @return string JSON-encoded array of MariaDB data with vulnerabilities and vulnerable status.
 */
function wpvulnerability_mariadb_get_installed() {

	$wpvulnerability_mariadb_vulnerable = 0;

	// Retrieve fresh vulnerabilities for the installed MariaDB version.
	$mariadb = wpvulnerability_get_fresh_mariadb_vulnerabilities();

	// Check if the MariaDB version is vulnerable and count the vulnerabilities.
	if ( isset( $mariadb['vulnerable'] ) && (int) $mariadb['vulnerable'] ) {
		$wpvulnerability_mariadb_vulnerable = count( $mariadb['vulnerabilities'] );
	}

	// Cache the vulnerability data and the timestamp for cache expiration.
	if ( is_multisite() ) {
		update_site_option( 'wpvulnerability-mariadb', wp_json_encode( $mariadb ) );
		update_site_option( 'wpvulnerability-mariadb-vulnerable', wp_json_encode( number_format( $wpvulnerability_mariadb_vulnerable, 0, '.', '' ) ) );
		update_site_option( 'wpvulnerability-mariadb-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	} else {
		update_option( 'wpvulnerability-mariadb', wp_json_encode( $mariadb ) );
		update_option( 'wpvulnerability-mariadb-vulnerable', wp_json_encode( number_format( $wpvulnerability_mariadb_vulnerable, 0, '.', '' ) ) );
		update_option( 'wpvulnerability-mariadb-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	}

	// Return the JSON-encoded array of MariaDB data.
	return wp_json_encode( $mariadb );
}

/**
 * Retrieve Cached MariaDB Vulnerabilities or Update the Cache if Stale.
 *
 * This function checks the cache for stored MariaDB vulnerabilities. If the cache is stale or missing,
 * it updates the cache with the latest vulnerabilities data.
 *
 * @since 3.4.0
 *
 * @return array Array of MariaDB data with their vulnerabilities.
 */
function wpvulnerability_mariadb_get_vulnerabilities() {

	if ( is_multisite() ) {

		// Get the cached MariaDB data expiration timestamp.
		$mariadb_data_cache = json_decode( get_site_option( 'wpvulnerability-mariadb-cache' ) );

		// Get the cached MariaDB data.
		$mariadb_data = json_decode( get_site_option( 'wpvulnerability-mariadb' ), true );

	} else {

		// Get the cached MariaDB data expiration timestamp.
		$mariadb_data_cache = json_decode( get_option( 'wpvulnerability-mariadb-cache' ) );

		// Get the cached MariaDB data.
		$mariadb_data = json_decode( get_option( 'wpvulnerability-mariadb' ), true );

	}

	// If the cache is stale or the MariaDB data is missing, update the cache.
	if ( $mariadb_data_cache < time() || empty( $mariadb_data ) ) {

		// Retrieve the latest MariaDB vulnerabilities and update the cache.
		$mariadb_data = json_decode( wpvulnerability_mariadb_get_installed(), true );

	}

	return $mariadb_data;
}

/**
 * Refresh the MariaDB Cache and Remove Stale Data.
 *
 * This function updates the cache for MariaDB vulnerabilities by retrieving the latest
 * installed MariaDB data. It ensures that the cache is refreshed with the most current
 * vulnerability information, effectively removing any outdated data.
 *
 * @since 3.4.0
 *
 * @return void
 */
function wpvulnerability_mariadb_get_vulnerabilities_clean() {

	// Refresh the cache by updating the installed MariaDB data.
	wpvulnerability_mariadb_get_installed();
}
