<?php
/**
 * Apache HTTPD functions
 *
 * @package WPVulnerability
 *
 * @version 3.2.0
 */

defined( 'ABSPATH' ) || die( 'No script kiddies please!' );

/**
 * Retrieves vulnerabilities for an Apache HTTPD version and updates its data.
 *
 * This function detects the installed Apache HTTPD version, checks for vulnerabilities using an external API,
 * and updates the Apache HTTPD data array with the vulnerabilities found.
 *
 * @since 3.2.0
 *
 * @return array The updated Apache HTTPD data array containing vulnerability information.
 */
function wpvulnerability_get_fresh_apache_vulnerabilities() {

	$apache_version = null;
	$webserver      = wpvulnerability_detect_webserver();
	if ( isset( $webserver['id'] ) && 'apache' === $webserver['id'] && isset( $webserver['version'] ) && $webserver['version'] ) {
		// Get the Apache HTTPD version.
		$apache_version = wp_kses( (string) $webserver['version'], 'strip' );
	}

	// Initialize vulnerability related fields.
	$apache_data['vulnerabilities'] = null;
	$apache_data['vulnerable']      = 0;

	// Retrieve vulnerabilities for the Apache HTTPD using its version.
	if ( $apache_version ) {

		$apache_api_response = wpvulnerability_get_apache( $apache_version, 0 );

		// If vulnerabilities are found, update the Apache HTTPD data accordingly.
		if ( ! empty( $apache_api_response ) ) {

			$apache_data['vulnerabilities'] = $apache_api_response;
			$apache_data['vulnerable']      = 1;

		}
	}

	return $apache_data;
}

/**
 * Get Installed Apache HTTPD
 *
 * Retrieves the list of installed Apache HTTPD versions, checks for vulnerabilities,
 * caches the data, and sends an email notification if vulnerabilities are detected.
 *
 * @since 3.2.0
 *
 * @return string JSON-encoded array of Apache HTTPD data with vulnerabilities and vulnerable status.
 */
function wpvulnerability_apache_get_installed() {

	$wpvulnerability_apache_vulnerable = 0;

	// Retrieve fresh vulnerabilities for the installed Apache HTTPD version.
	$apache = wpvulnerability_get_fresh_apache_vulnerabilities();

	// Check if the Apache HTTPD version is vulnerable and count the vulnerabilities.
	if ( isset( $apache['vulnerable'] ) && (int) $apache['vulnerable'] ) {
		$wpvulnerability_apache_vulnerable = count( $apache['vulnerabilities'] );
	}

	// Cache the vulnerability data and the timestamp for cache expiration.
	if ( is_multisite() ) {
		update_site_option( 'wpvulnerability-apache', wp_json_encode( $apache ) );
		update_site_option( 'wpvulnerability-apache-vulnerable', wp_json_encode( number_format( $wpvulnerability_apache_vulnerable, 0, '.', '' ) ) );
		update_site_option( 'wpvulnerability-apache-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	} else {
		update_option( 'wpvulnerability-apache', wp_json_encode( $apache ) );
		update_option( 'wpvulnerability-apache-vulnerable', wp_json_encode( number_format( $wpvulnerability_apache_vulnerable, 0, '.', '' ) ) );
		update_option( 'wpvulnerability-apache-cache', wp_json_encode( number_format( time() + ( 3600 * WPVULNERABILITY_CACHE_HOURS ), 0, '.', '' ) ) );
	}

	// Return the JSON-encoded array of Apache HTTPD data.
	return wp_json_encode( $apache );
}

/**
 * Get the cached Apache HTTPD vulnerabilities or update the cache if it's stale or missing.
 *
 * This function checks the cache for stored Apache HTTPD vulnerabilities. If the cache is stale or missing,
 * it updates the cache with the latest vulnerabilities data.
 *
 * @since 3.2.0
 *
 * @return array Array of Apache HTTPD data with their vulnerabilities.
 */
function wpvulnerability_apache_get_vulnerabilities() {

	if ( is_multisite() ) {

		// Get the cached Apache HTTPD data and decode it.
		$apache_data_cache = json_decode( get_site_option( 'wpvulnerability-apache-cache' ) );

		// Get the installed Apache HTTPD data and decode it.
		$apache_data = json_decode( get_site_option( 'wpvulnerability-apache' ), true );

	} else {

		// Get the cached Apache HTTPD data and decode it.
		$apache_data_cache = json_decode( get_option( 'wpvulnerability-apache-cache' ) );

		// Get the installed Apache HTTPD data and decode it.
		$apache_data = json_decode( get_option( 'wpvulnerability-apache' ), true );

	}

	// If the cache is stale or the Apache HTTPD data is empty, update the cache.
	if ( $apache_data_cache < time() || empty( $apache_data ) ) {

		// Get the installed Apache HTTPD data and update the cache.
		$apache_data = json_decode( wpvulnerability_apache_get_installed(), true );

	}

	return $apache_data;
}

/**
 * Update the Apache HTTPD cache and remove any old cache data.
 *
 * This function refreshes the cache for Apache HTTPD vulnerabilities by calling the function
 * that retrieves and updates the installed Apache HTTPD data, ensuring that the cache contains
 * the most recent information.
 *
 * @since 3.2.0
 *
 * @return void
 */
function wpvulnerability_apache_get_vulnerabilities_clean() {

	// Update the installed Apache HTTPD cache.
	wpvulnerability_apache_get_installed();
}
